/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.json.jose.jwk.store;

import java.net.URL;
import java.util.concurrent.TimeUnit;
import org.forgerock.json.jose.exceptions.FailedToLoadJWKException;
import org.forgerock.json.jose.jwk.JWK;
import org.forgerock.json.jose.jwk.JWKSet;
import org.forgerock.json.jose.jwk.JWKSetParser;
import org.forgerock.json.jose.jwk.KeyUse;
import org.forgerock.json.jose.jwt.Algorithm;
import org.forgerock.util.Reject;
import org.forgerock.util.SimpleHTTPClient;
import org.forgerock.util.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwksStore {
    private static final Logger logger = LoggerFactory.getLogger(JwksStore.class);
    private final JWKSetParser jwkParser;
    private long cacheMissCacheTimeInMs;
    private long cacheTimeoutInMs;
    private URL jwkUrl;
    private JWKSet jwksSet;
    private long lastReloadJwksSet;

    JwksStore(Duration cacheTimeout, Duration cacheMissCacheTime, URL jwkUrl, SimpleHTTPClient httpClient) throws FailedToLoadJWKException {
        this(cacheTimeout, cacheMissCacheTime, jwkUrl, new JWKSetParser(httpClient));
    }

    JwksStore(Duration cacheTimeout, Duration cacheMissCacheTime, URL jwkUrl, JWKSetParser jwkSetParser) throws FailedToLoadJWKException {
        this.cacheTimeoutInMs = cacheTimeout.to(TimeUnit.MILLISECONDS);
        this.cacheMissCacheTimeInMs = cacheMissCacheTime.to(TimeUnit.MILLISECONDS);
        this.jwkUrl = jwkUrl;
        this.jwkParser = jwkSetParser;
        try {
            this.reloadJwks();
        }
        catch (FailedToLoadJWKException e) {
            logger.debug("Unable to load keys from the JWK over HTTP");
            throw new FailedToLoadJWKException("Unable to load keys from the JWK over HTTP", e);
        }
    }

    private synchronized void reloadJwks() throws FailedToLoadJWKException {
        this.jwksSet = this.jwkParser.jwkSet(this.jwkUrl);
        this.lastReloadJwksSet = System.currentTimeMillis();
    }

    public JWK findJwk(Algorithm algorithm, KeyUse keyUse) throws FailedToLoadJWKException {
        JWK jwk;
        if (keyUse == KeyUse.ENC && this.hasJwksCacheTimedOut()) {
            this.reloadJwks();
        }
        if ((jwk = this.jwksSet.findJwk(algorithm, keyUse)) == null && this.isCacheMissCacheTimeExpired()) {
            this.reloadJwks();
            return this.jwksSet.findJwk(algorithm, keyUse);
        }
        return jwk;
    }

    public JWK findJwk(String kid) throws FailedToLoadJWKException {
        JWK jwk = this.jwksSet.findJwk(kid);
        if (jwk == null && this.isCacheMissCacheTimeExpired()) {
            this.reloadJwks();
            return this.jwksSet.findJwk(kid);
        }
        return jwk;
    }

    public Duration getCacheTimeout() {
        return Duration.duration(this.cacheTimeoutInMs, TimeUnit.MILLISECONDS);
    }

    public Duration getCacheMissCacheTime() {
        return Duration.duration(this.cacheMissCacheTimeInMs, TimeUnit.MILLISECONDS);
    }

    public URL getJwkUrl() {
        return this.jwkUrl;
    }

    public void setCacheTimeout(Duration cacheTimeout) {
        this.cacheTimeoutInMs = cacheTimeout.to(TimeUnit.MILLISECONDS);
    }

    public void setCacheMissCacheTime(Duration cacheMissCacheTime) {
        this.cacheMissCacheTimeInMs = cacheMissCacheTime.to(TimeUnit.MILLISECONDS);
    }

    public void setJwkUrl(URL jwkUrl) throws FailedToLoadJWKException {
        Reject.ifNull(jwkUrl);
        URL originalJwkUrl = this.jwkUrl;
        this.jwkUrl = jwkUrl;
        if (!jwkUrl.equals(originalJwkUrl)) {
            this.reloadJwks();
        }
    }

    private boolean hasJwksCacheTimedOut() {
        return System.currentTimeMillis() - this.lastReloadJwksSet > this.cacheTimeoutInMs;
    }

    private boolean isCacheMissCacheTimeExpired() {
        return System.currentTimeMillis() - this.lastReloadJwksSet >= this.cacheMissCacheTimeInMs;
    }
}

