package com.inwebo.demo_android.network;

import android.util.Log;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * PromiseSupport was created to handle asynchronous computations using Future interface
 * A Promise represents an operation that hasn't completed yet, but is expected in the future.
 * The PromiseSupport only handles status and completions
 * @param <T> the object to communicate through the promise.
 */
public class PromiseSupport<T> implements Future<T> {
    private static final int RUNNING = 1;
    private static final int FAILED = 2;
    private static final int COMPLETED = 3;

    private final Object lock;

    private volatile int state = RUNNING;
    private T value;
    private Exception exception;

    PromiseSupport() {
        this.lock = new Object();
    }

    void fulfill(T value) {
        this.value = value;
        this.state = COMPLETED;
        synchronized (lock) {
            lock.notifyAll();
        }
    }

    void fulfillExceptionally(Exception exception) {
        this.exception = exception;
        this.state = FAILED;
        synchronized (lock) {
            lock.notifyAll();
        }
    }

    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
        return false;
    }

    @Override
    public boolean isCancelled() {
        return false;
    }

    @Override
    public boolean isDone() {
        return state > RUNNING;
    }

    @Override
    public T get() throws InterruptedException, ExecutionException {
        synchronized (lock) {
            while (state == RUNNING) {
                lock.wait();
            }
        }
        if (state == COMPLETED) {
            return value;
        }
        throw new ExecutionException(exception);
    }

    @Override
    public T get(long timeout, TimeUnit unit) throws ExecutionException {
        synchronized (lock) {
            while (state == RUNNING) {
                try {
                    lock.wait(unit.toMillis(timeout));
                } catch (InterruptedException e) {
                    Log.w("Interrupted!", e);
                    Thread.currentThread().interrupt();
                }
            }
        }

        if (state == COMPLETED) {
            return value;
        }
        throw new ExecutionException(exception);
    }
}