package com.inwebo.demo_android;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

/**
 * OnlineOtpGeneratorActivity
 */
public class OnlineOtpGeneratorActivity extends InweboActivity {
    private static final String TAG = OnlineOtpGeneratorActivity.class.getSimpleName();

    // containers
    private LinearLayout otpGeneratedContainer;

    // inputs
    private TextView otpGeneratorPinTitleView;
    private EditText otpGeneratorPinInput;
    private Button otpGeneratorButton;
    private TextView otpGeneratedInput;
    private Button otpGeneratorCopyButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_online_otp_generator);

        this.otpGeneratedContainer = this.findViewById(R.id.otp_generator_response_container);
        this.otpGeneratedContainer = this.findViewById(R.id.otp_generator_response_container);

        this.otpGeneratorPinTitleView = this.findViewById(R.id.otp_generator_pin_title);
        this.otpGeneratorPinInput = this.findViewById(R.id.otp_generator_pin_input);
        this.otpGeneratorButton = this.findViewById(R.id.otp_generator_button);
        this.otpGeneratedInput = this.findViewById(R.id.otp_generator_response);
        this.otpGeneratorCopyButton = this.findViewById(R.id.otp_generator_copy_button);
    }

    @Override
    protected void onStart() {
        super.onStart();
        this.otpGeneratorPinInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.otpGeneratorButton)
        );
        this.otpGeneratorButton.setOnClickListener((v) -> {
            InweboUiUtils.hideKeyboard(this, v);
            this.startOnlineOtpGenerationFinalize();
        });
        this.otpGeneratorCopyButton.setOnClickListener((v) -> {
            ClipboardManager clipboard = (ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            ClipData clip = ClipData.newPlainText("generated_otp", this.otpGeneratedInput.getText().toString());
            clipboard.setPrimaryClip(clip);
            this.otpGeneratorCopyButton.setClickable(false);

            Toast.makeText(
                    this,
                    this.getResources().getString(R.string.otp_generator_copy_successful),
                    Toast.LENGTH_SHORT
            ).show();
        });
        this.startOnlineOtpGenerationProcess();
    }

    private void startOnlineOtpGenerationProcess() {
        Log.i(TAG, "Starting Online OTP Generation process");
        this.resetUi();
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().OnlineOtpStart(0))
                .thenAccept(responseId -> {
                    Log.d(TAG, "ActivationFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.handleOnlineOtpGeneration();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.finish();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void handleOnlineOtpGeneration() {
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "PIN mode is " + pinMode);

        if ((pinMode & IW.PINMODE_CURRENT) != 0) {
            this.toggleInputVisibility(true);
            this.toggleInputAccessibility(true);
        }
    }

    private void startOnlineOtpGenerationFinalize() {
        this.toggleInputAccessibility(false);

        String pin = this.otpGeneratorPinInput.getText().toString();
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "PIN mode is " + pinMode);

        if (pinMode == IW.PINMODE_NONE) {
            this.handleOnlineOtpGenerationFinalize("");
        } else if ((pinMode & IW.PINMODE_CURRENT) != 0) {
            if (InweboUtils.isEmpty(pin)) {
                Log.d(TAG, "Activation code value is empty");
                InweboUiUtils.showErrorDialog(this, R.string.pin_empty);
                this.startOnlineOtpGenerationProcess();
                return;
            }
            this.handleOnlineOtpGenerationFinalize(pin);
        }
    }

    private void handleOnlineOtpGenerationFinalize(String pin) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().OnlineOtpFinalize(0, pin))
                .thenAccept(responseId -> {
                    Log.d(TAG, "ActivationFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        String otpGenerated = super.inweboService.getApi().OtpAnswerOtp();
                        if (InweboUtils.isEmpty(otpGenerated)) {
                            Log.e(TAG, "Generated OTP is empty");
                            InweboUiUtils.showErrorDialog(this, R.string.otp_generator_error);
                            this.startOnlineOtpGenerationProcess();
                            return;
                        }
                        this.otpGeneratedContainer.setVisibility(View.VISIBLE);
                        this.otpGeneratedInput.setText(otpGenerated);
                        this.otpGeneratedInput.setEnabled(false);

                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.startOnlineOtpGenerationProcess();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void toggleInputVisibility(boolean isVisible) {
        int visibility = isVisible ? View.VISIBLE : View.INVISIBLE;
        this.otpGeneratorPinTitleView.setVisibility(visibility);
        this.otpGeneratorPinInput.setVisibility(visibility);
        this.otpGeneratorPinInput.setText("");
    }

    private void toggleInputAccessibility(boolean isEnabled) {
        this.otpGeneratorPinInput.setEnabled(isEnabled);
        this.otpGeneratorButton.setClickable(isEnabled);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.toggleInputAccessibility(false);
        this.toggleInputVisibility(false);
        this.otpGeneratedContainer.setVisibility(View.GONE);
        // this button will be hidden anyway
        this.otpGeneratorCopyButton.setClickable(true);
    }
}