package com.inwebo.demo_android;

import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.ArrayAdapter;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

import com.inwebo.demo_android.service.biometric.BiometricService;
import com.inwebo.demo_android.ui.MenuAdapter;
import com.inwebo.demo_android.ui.MenuItem;

import java.util.ArrayList;
import java.util.List;

/**
 * MenuActivity is the main activity of this application
 * It checks the inwebo status when starting, and redirect to
 * -> the FirstConnection Activity if the user is not registered
 * -> the Unlock Activity if the user is blocked
 * -> the menu if the user is correctly registered
 */
public class MenuActivity extends InweboActivity {
    private static final String TAG = MenuActivity.class.getSimpleName();

    private BiometricService biometricService;

    // containers
    protected LinearLayout activationContainer;
    protected LinearLayout pinActivationContainer;
    protected LinearLayout serviceContainer;
    protected LinearLayout messageContainer;

    // views
    private ListView serviceListView;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_menu);

        this.activationContainer = this.findViewById(R.id.activation_container);
        this.pinActivationContainer = this.findViewById(R.id.activation_pin_container);
        this.serviceContainer = this.findViewById(R.id.services_container);
        this.messageContainer = this.findViewById(R.id.message_container);

        this.serviceListView = this.findViewById(R.id.list_of_services_view);

        this.biometricService = BiometricService.with(this);
    }

    @Override
    protected void onStart() {
        super.onStart();
        checkInweboStatus();
    }

    public void checkInweboStatus() {
        Log.i(TAG, "Checking Inwebo status");
        // isBlocked is true if the user is blocked
        if (this.inweboService.isBlocked()) {
            Log.d(TAG, "User is blocked");
            Intent unlockActivity = new Intent(this, UnlockActivity.class);
            this.startActivity(unlockActivity);
            return;
        }
        // shouldActivate is true if the user is not registered
        if (this.inweboService.shouldActivate()) {
            Log.d(TAG, "User is not registered");
            this.serviceContainer.setVisibility(View.GONE);
            this.activationContainer.setVisibility(View.VISIBLE);
            this.pinActivationContainer.setVisibility(View.GONE);
            return;
        }
        this.displayServices();
    }

    public void displayServices() {
        Log.d(TAG, "Displaying services");
        this.serviceContainer.setVisibility(View.VISIBLE);
        this.activationContainer.setVisibility(View.GONE);

        List<MenuItem> services = new ArrayList<>();
        services.add(MenuItem.of(
                this.getResources().getString(R.string.synchronize),
                SynchronizeActivity.class
        ));
        services.add(MenuItem.of(
                this.getResources().getString(R.string.change_password),
                PasswordUpdateActivity.class
        ));
        services.add(MenuItem.of(
                this.getResources().getString(R.string.unlock_device),
                UnlockActivity.class
        ));
        services.add(MenuItem.of(
                this.getResources().getString(R.string.online_otp),
                OnlineOtpGeneratorActivity.class
        ));
        if (this.biometricService.canAuthenticateWithBioKey()) {
            services.add(MenuItem.of(
                    this.getResources().getString(R.string.bio_online_otp),
                    BiometricOtpGeneratorActivity.class
            ));
            services.add(MenuItem.of(
                    this.getResources().getString(R.string.bio_register),
                    BiometricRegisterActivity.class
            ));
        }
        services.add(MenuItem.of(
                this.getResources().getString(R.string.offline_otp),
                OfflineOtpGeneratorActivity.class
        ));

        services.add(MenuItem.of(
                this.getResources().getString(R.string.online_seal),
                OnlineSealActivity.class
        ));

        services.add(MenuItem.of(
                this.getResources().getString(R.string.offline_seal),
                OfflineSealActivity.class
        ));

        services.add(MenuItem.of(
                this.getResources().getString(R.string.pending_push),
                PendingPushActivity.class
        ));

        ArrayAdapter<MenuItem> adapter = new MenuAdapter(services, MenuActivity.this);

        this.serviceListView.setAdapter(adapter);
    }
}