package com.inwebo.demo_android;

import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

import androidx.annotation.NonNull;

/**
 * FirstConnectionActivity is the launched activity when the application is started
 * It will first call the MenuActivity that checks if the user needs to authenticate
 * If the user is already authenticate, then the menu will appear
 * Else, the authentication form will appear and follow this flow
 */
public class FirstConnectionActivity extends MenuActivity {
    private static final String TAG = FirstConnectionActivity.class.getSimpleName();

    // actions
    private EditText activationCodeInput;
    private Button activationCodeButton;

    private TextView activationPinTitle;
    private EditText activationPinInput;
    private EditText activationPinConfirmationInput;
    private Button activationPinButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        this.activationCodeInput = this.findViewById(R.id.activation_code_input);
        this.activationCodeButton = this.findViewById(R.id.activation_code_button);

        this.activationPinTitle = this.findViewById(R.id.activation_pin_title);
        this.activationPinInput = this.findViewById(R.id.activation_pin_input);
        this.activationPinConfirmationInput = this.findViewById(R.id.activation_pin_confirmation_input);
        this.activationPinButton = this.findViewById(R.id.activation_pin_button);
    }

    @Override
    protected void onStart() {
        super.onStart();
        this.activationPinButton.setOnClickListener(view -> {
            InweboUiUtils.hideKeyboard(this, view);
            validatePinCode();
        });

        this.activationCodeInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.activationCodeButton)
        );
        this.activationCodeButton.setOnClickListener(v -> {
            InweboUiUtils.hideKeyboard(this, v);
            validateCode();
        });

        this.activationPinConfirmationInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.activationPinButton)
        );
    }

    public void startActivationProcess(String activationCodeValue) {
        Log.i(TAG, "Starting Activation process");
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().ActivationStart(activationCodeValue))
                .thenAccept(responseId -> {
                    Log.d(TAG, "ActivationStart responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.handleActivationProcess(activationCodeValue);
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.toggleActivationCodeAccessibility(true);
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    public void handleActivationProcess(String codeValue) {
        long pinMode = super.inweboService.getApi().PinMode();
        this.pinActivationContainer.setVisibility(View.VISIBLE);

        Log.d(TAG, "Pin Mode is " + pinMode);
        if (pinMode == IW.PINMODE_NONE) {
            // Pin Mode is NONE when no password is required
            this.pinActivationContainer.setVisibility(View.GONE);
            this.startActivationFinalize(codeValue, "");
        } else if (pinMode == IW.PINMODE_CURRENT) {
            // Pin Mode is CURRENT when current password is required
            this.pinActivationContainer.setVisibility(View.VISIBLE);
            this.activationPinConfirmationInput.setVisibility(View.GONE);
        } else {
            this.activationPinTitle.setText(R.string.activation_define_pin_text);
            this.pinActivationContainer.setVisibility(View.VISIBLE);
        }
    }

    public void startActivationFinalize(String codeValue, String pinValue) {
        Log.i(TAG, "Starting Activation Finalize process");
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().ActivationFinalize(
                        codeValue,
                        pinValue,
                        InweboUtils.phoneName()
                ))
                .thenAccept(responseId -> {
                    Log.d(TAG, "ActivationFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);

                    if (responseId == IW.ERR_OK) {
                        Toast.makeText(
                                getApplicationContext(),
                                this.getResources().getString(R.string.activation_successful),
                                Toast.LENGTH_SHORT
                        ).show();

                        super.displayServices();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.resetUi();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    public void validateCode() {
        String activationCodeValue = this.activationCodeInput.getText().toString();

        this.toggleActivationCodeAccessibility(false);

        if (InweboUtils.isEmpty(activationCodeValue)) {
            Log.d(TAG, "Activation code value is empty");
            InweboUiUtils.showErrorDialog(this, R.string.code_empty);

            this.toggleActivationCodeAccessibility(true);
            return;
        }
        this.startActivationProcess(activationCodeValue);
    }

    public void validatePinCode() {
        String codeValue = this.activationCodeInput.getText().toString();
        String pinCodeValue = this.activationPinInput.getText().toString();
        String pinConfirmValue = this.activationPinConfirmationInput.getText().toString();

        this.toggleActivationPinAccessibility(false);

        if (InweboUtils.isEmpty(pinCodeValue)) {
            Log.d(TAG, "PIN code value is empty");
            InweboUiUtils.showErrorDialog(this, R.string.pin_empty);

            this.toggleActivationPinAccessibility(true);
            return;
        } else if (super.inweboService.getApi().PinMode() == IW.PINMODE_NEW
                && !pinCodeValue.equals(pinConfirmValue)
        ) {
            Log.d(TAG, "PIN code value is not the same than the confirmation PIN code value");
            InweboUiUtils.showErrorDialog(this, R.string.pin_confirmation_different);

            this.toggleActivationPinAccessibility(true);
            return;
        } else if (!InweboUtils.isPinValid(pinCodeValue)) {
            Log.d(TAG, "PIN code value is not valid");
            InweboUiUtils.showErrorDialog(this, R.string.pin_wrong_format);

            this.toggleActivationPinAccessibility(true);
            return;
        }
        this.startActivationFinalize(codeValue, pinCodeValue);
    }

    public void toggleActivationCodeAccessibility(boolean isEnable) {
        this.activationCodeInput.setEnabled(isEnable);
        this.activationCodeButton.setClickable(isEnable);
    }

    public void toggleActivationPinAccessibility(boolean isEnable) {
        this.activationPinInput.setEnabled(isEnable);
        this.activationPinConfirmationInput.setEnabled(isEnable);
        this.activationPinButton.setClickable(isEnable);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.activationCodeInput.setText("");
        this.activationPinInput.setText("");
        this.activationPinConfirmationInput.setText("");
        this.toggleActivationCodeAccessibility(true);
        this.toggleActivationPinAccessibility(true);
        this.pinActivationContainer.setVisibility(View.GONE);
    }
}
