package com.inwebo.demo_android.service;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.util.Base64;
import android.util.Log;

import com.inwebo.demo_android.BuildConfig;
import com.inwebo.demo_android.R;
import com.inwebo.demo_android.service.cryptography.CryptographyException;
import com.inwebo.demo_android.service.cryptography.CryptographyManager;
import com.inwebo.demo_android.service.cryptography.EncryptedData;
import com.inwebo.demo_android.utils.FileUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

import javax.crypto.Cipher;

import static android.content.Context.MODE_PRIVATE;

/**
 * InweboService is the service that handles and the communications with the inWebo library
 * The with(Activity) method ensure that this service is only instantiated once
 * The constructor initialises the inWebo API to be able to communicate with
 */
public class InweboService {
    private static final String TAG = InweboService.class.getSimpleName();
    private static final String DATA_FILE_NAME = "data";
    private static final String SHARED_PREFERENCE_KEY = "demo_android";
    private static final String SHARED_PREFERENCE_DATA_KEY_IV = "encrypted_data_iv";

    private static final String VERSION = BuildConfig.VERSION_NAME;
    private static final String MAC_ID = BuildConfig.MACID;
    private static final String SERVER = BuildConfig.SERVER;
    private static final int TIMEOUT_NETWORK = 60000;

    private static InweboService INSTANCE;

    private final IW iw;
    private final CryptographyManager cryptographyManager;
    private final SharedPreferences sharedPreferences;

    /**
     * get the singleton instance
     *
     * @param activity : the activity where the session is called
     * @return the session singleton
     */
    public static InweboService with(Activity activity) {
        // first check to see if instance is already created
        if (INSTANCE == null) {
            // synchronized call to prevent from multi thread
            synchronized (InweboService.class) {
                // second check to see if instance was created with an other concurrent thread
                if (INSTANCE == null) {
                    INSTANCE = new InweboService(activity);
                }
            }
        }
        return INSTANCE;
    }

    private InweboService(Activity activity) {
        this.iw = new IW();
        this.iw.Init(InweboUtils.phoneUniqueId(activity), "_");

        Log.i(TAG, "Init inWebo with config SERVER : " + SERVER);
        Log.i(TAG, "Init inWebo with config MAC_ID " + MAC_ID);

        this.iw.WsServerSet(SERVER);
        this.iw.HostVersionSet("demo_android-" + VERSION);
        this.iw.WsTimeoutSet(TIMEOUT_NETWORK);
        this.iw.MaccessSet(MAC_ID);

        this.cryptographyManager = CryptographyManager.getInstance();
        this.sharedPreferences = activity.getSharedPreferences(SHARED_PREFERENCE_KEY, MODE_PRIVATE);

        setDataFromInternalStorage(activity);
    }

    public IW getApi() {
        return iw;
    }

    private String getInternalData(Context context) {
        String encryptedDataString = FileUtils.readFile(context, DATA_FILE_NAME);
        if (encryptedDataString == null) {
            return "";
        }
        byte[] encryptedDataIv = Base64.decode(this.sharedPreferences.getString(SHARED_PREFERENCE_DATA_KEY_IV, ""), Base64.NO_WRAP);
        byte[] encryptedData = Base64.decode(encryptedDataString, Base64.NO_WRAP);
        Cipher cipher = this.cryptographyManager.getInitializedCipherForDecryption(
                context.getString(R.string.inwebo_data_secret_key),
                encryptedDataIv,
                false
        );

        if (encryptedData != null) {
            return this.handleDataDecryption(context, encryptedData, cipher);
        }

        return "";
    }

    public void setDataFromInternalStorage(Activity activity) {
        Log.i(TAG, "setDataFromInternalStorage");
        String data = getInternalData(activity);
        iw.StorageDataSet(data);
    }

    public void setDataToInternalStorage(Context context) {
        Log.i(TAG, "setDataToInternalStorage");
        if (iw.StorageDataChanged() > 0) {
            String data = iw.StorageDataGet();
            Cipher cipher = this.cryptographyManager.getInitializedCipherForEncryption(
                    context.getString(R.string.inwebo_data_secret_key),
                    false
            );
            this.handleDataEncryption(context, data, cipher);
            return;
        }
        Log.i(TAG, "setDataToInternalStorage : nothing changed");
    }

    public boolean shouldActivate() {
        boolean shouldActivate = iw.IsActivated() != 1;
        Log.i(TAG, "shouldActivate is " + shouldActivate);
        return shouldActivate;
    }

    public void setDeviceOSTest() {
        iw.SetDeviceOS("iwtest");
    }

    public boolean sealShouldSynchronize() {
        boolean shouldSynchronize = iw.SealShouldSynchronize(0) == 1;
        Log.i(TAG, "shouldSynchronize is " + shouldSynchronize);
        return shouldSynchronize;
    }

    public boolean otpShouldSynchronize() {
        boolean shouldSynchronize = iw.OtpShouldSynchronize(0) == 1;
        Log.i(TAG, "shouldSynchronize is " + shouldSynchronize);
        return shouldSynchronize;
    }

    public boolean isBlocked() {
        int localBlocked = (int) iw.IsBlocked();
        //App/Device locally locked
        if (0 != localBlocked) {
            //Check is App/Device also locked server side
            int serverBlocked = (int) iw.CheckStatus();
            return serverBlocked != 0;
        }
        return false;
    }

    private void handleDataEncryption(Context context, String data, Cipher cipher) {
        try {
            EncryptedData encryptedData = this.cryptographyManager.encryptData(data, cipher);
            SharedPreferences.Editor editor = this.sharedPreferences.edit();
            editor.putString(
                    SHARED_PREFERENCE_DATA_KEY_IV,
                    Base64.encodeToString(encryptedData.getInitialisationVector(), Base64.NO_WRAP)
            );
            editor.apply();

            FileUtils.writeData(
                    context,
                    DATA_FILE_NAME,
                    Base64.encodeToString(encryptedData.getCipherText(), Base64.NO_WRAP)
            );
        } catch (CryptographyException e) {
            Log.e(TAG, "Can't encrypt data", e);
        }
    }

    private String handleDataDecryption(Context context, byte[] encryptedData, Cipher cipher) {
        try {
            return this.cryptographyManager.decryptData(encryptedData, cipher);
        } catch (CryptographyException e) {
            SharedPreferences.Editor edit = this.sharedPreferences.edit();
            edit.remove(SHARED_PREFERENCE_DATA_KEY_IV);
            edit.apply();
            context.deleteFile(DATA_FILE_NAME);
            Log.e(TAG, "Can't decrypt stored data. Application data were flushed.", e);
        }
        return "";
    }
}
