package com.inwebo.demo_android;

import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;
import com.google.firebase.messaging.FirebaseMessaging;
import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

import androidx.annotation.NonNull;

/**
 * PendingPushActivity implements the push registration process and check pending push process
 * The push registration is required to use the check pending push service
 * DeviceOS is set to "iwtest" to use push services in test mode
 */
public class PendingPushActivity extends InweboActivity {
    private static final String TAG = PendingPushActivity.class.getSimpleName();

    // containers
    private LinearLayout pendingPushResultContainer;

    // inputs
    private Button pendingPushButton;
    private Button registerPushButton;
    private TextView pendingPushAlias;
    private TextView pendingPushContext;
    private TextView pendingPushAction;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_pending_push);

        this.pendingPushResultContainer = this.findViewById(R.id.pending_push_result_container);
        this.pendingPushButton = this.findViewById(R.id.pending_push_button);
        this.registerPushButton = this.findViewById(R.id.register_push_button);
        this.pendingPushAlias = this.findViewById(R.id.pending_push_alias_result);
        this.pendingPushContext = this.findViewById(R.id.pending_push_context_result);
        this.pendingPushAction = this.findViewById(R.id.pending_push_action_result);
    }

    @Override
    protected void onStart() {
        super.onStart();
        this.pendingPushResultContainer.setVisibility(View.GONE);

        this.pendingPushButton.setOnClickListener((v) -> {
            this.startCheckPendingPush();
        });

        this.registerPushButton.setOnClickListener((v) -> {
            this.registerForPushNotification();
        });
    }

    private void registerForPushNotification() {
        FirebaseMessaging.getInstance().getToken()
                .addOnCompleteListener(task -> {
                    if (!task.isSuccessful()) {
                        Log.w(TAG, "Fetching FCM registration token failed", task.getException());
                        return;
                    }

                    // Get new FCM registration token
                    final String token = task.getResult();

                    PendingPushActivity.this.inweboService.getApi().SetDeviceOS("firebase");

                    Log.i(TAG, "Starting push registration");
                    PendingPushActivity.this.resetUi();
                    PendingPushActivity.this.toggleCheckButtonAccessibility(false);
                    InweboPromise.with(PendingPushActivity.this)
                            .fulfillInAsync(() ->
                                    PendingPushActivity.this.inweboService.getApi().PushRegistrationStart()
                            )
                            .thenAccept(responseId -> {
                                Log.d(TAG, "PushRegistrationStart responded with code " + responseId);
                                PendingPushActivity.this.inweboService.setDataToInternalStorage(PendingPushActivity.this);
                                if (responseId == IW.ERR_OK) {
                                    finalizePushRegistration(token);
                                    return;
                                }
                                InweboUiUtils.showErrorDialogFromResponseId(PendingPushActivity.this, responseId);
                                PendingPushActivity.this.finish();
                            })
                            .onError(throwable -> {
                                Log.e(TAG, "Error during promise execution", throwable);
                                InweboUiUtils.showNetworkErrorDialog(this);
                            });
                });
    }

    private void finalizePushRegistration(String fcmToken) {
        Log.i(TAG, "Finalize push registration");
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().PushRegistrationFinalize(fcmToken))
                .thenAccept(responseId -> {
                    Log.d(TAG, "PushRegistrationFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.toggleCheckButtonAccessibility(true);
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.finish();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void startCheckPendingPush() {
        Log.i(TAG, "Starting check for pending push");
        this.toggleCheckButtonAccessibility(false);
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().CheckPush())
                .thenAccept(responseId -> {
                    Log.d(TAG, "PendingPush responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    this.toggleCheckButtonAccessibility(true);
                    if (responseId == IW.ERR_OK) {
                        String alias = this.getPushAlias();
                        String action = this.getPushAction();

                        this.pendingPushAlias.setText(alias);
                        this.pendingPushContext.setText(this.getPushContext());
                        this.pendingPushAction.setText(action);
                        this.pendingPushResultContainer.setVisibility(View.VISIBLE);

                        if ("authenticate".equals(action)) {
                            showPushOTPAuthentication(null, alias);
                        }
                        return;
                    }

                    if (responseId == IW.ERR_OTHER) {
                        InweboUiUtils.showErrorDialog(this, R.string.check_pending_push_no_push_activation);
                        this.resetUi();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.resetUi();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private String getPushAlias() {
        Log.i(TAG, "Get push alias");
        return super.inweboService.getApi().PushAlias();
    }

    private String getPushContext() {
        Log.i(TAG, "Get push context");
        return super.inweboService.getApi().PushContext();
    }

    private String getPushAction() {
        Log.i(TAG, "Get push action");
        return super.inweboService.getApi().PushAction();
    }

    private void toggleCheckButtonAccessibility(boolean isEnabled) {
        this.pendingPushButton.setEnabled(isEnabled);
        this.pendingPushButton.setClickable(isEnabled);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.pendingPushAlias.setText("");
        this.pendingPushContext.setText("");
        this.pendingPushAction.setText("");
        this.pendingPushResultContainer.setVisibility(View.GONE);
    }
}
