package com.inwebo.demo_android;

import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

/**
 * PasswordUpdateActivity implements the password update process
 */
public class PasswordUpdateActivity extends InweboActivity {
    private static final String TAG = PasswordUpdateActivity.class.getSimpleName();

    // inputs
    private TextView passwordUpdateCurrentPasswordTitleView;
    private TextView passwordUpdateNewPasswordTitleView;
    private EditText passwordUpdateCurrentPinInput;
    private EditText passwordUpdateNewPinInput;
    private EditText passwordUpdateNewPinConfirmationInput;
    private Button passwordUpdateButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_password_update);

        this.passwordUpdateCurrentPasswordTitleView = this.findViewById(R.id.password_update_current_password_title_view);
        this.passwordUpdateNewPasswordTitleView = this.findViewById(R.id.password_update_new_password_title_view);
        this.passwordUpdateCurrentPinInput = this.findViewById(R.id.password_update_current_password);
        this.passwordUpdateNewPinInput = this.findViewById(R.id.password_update_new_password_input);
        this.passwordUpdateNewPinConfirmationInput = this.findViewById(R.id.password_update_new_password_input_confirmation);
        this.passwordUpdateButton = this.findViewById(R.id.password_update_pin_button);
    }

    @Override
    protected void onStart() {
        super.onStart();

        this.passwordUpdateNewPinConfirmationInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.passwordUpdateButton)
        );
        this.passwordUpdateButton.setOnClickListener(view -> {
            InweboUiUtils.hideKeyboard(this, view);
            startPasswordUpdateFinalize();
        });

        this.startPasswordUpdateProcess();
    }

    private void startPasswordUpdateProcess() {
        Log.i(TAG, "Starting Password Update process");
        this.resetUI();
        InweboPromise.with(this)
                .fulfillInAsync(() -> this.inweboService.getApi().PwdUpdateStart())
                .thenAccept(responseId -> {
                    Log.d(TAG, "PwdUpdateStart responded with code " + responseId);
                    if (responseId == IW.ERR_OK) {
                        this.inweboService.setDataToInternalStorage(this);
                        this.handleStartPasswordProcess();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.finish();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void startPasswordUpdateFinalize() {
        Log.i(TAG, "Starting Password Update Finalize process");
        this.togglePinInputAccessibility(false);

        String currentPinValue = this.passwordUpdateCurrentPinInput.getText().toString();
        String newPinValue = this.passwordUpdateNewPinInput.getText().toString();
        String newPinConfirmValue = this.passwordUpdateNewPinConfirmationInput.getText().toString();

        if (InweboUtils.isEmpty(currentPinValue)
                || InweboUtils.isEmpty(newPinValue)
                || InweboUtils.isEmpty(newPinConfirmValue)
        ) {
            Log.d(TAG, "PIN value is empty");
            InweboUiUtils.showErrorDialog(this, R.string.pin_empty);
            this.togglePinInputAccessibility(true);
            return;
        } else if (!newPinValue.equals(newPinConfirmValue)) {
            Log.d(TAG, "PIN value is not the same as PIN confirmation values");
            InweboUiUtils.showErrorDialog(this, R.string.pin_confirmation_different);
            this.togglePinInputAccessibility(true);
            return;
        } else if (!InweboUtils.isPinValid(newPinConfirmValue)) {
            Log.d(TAG, "PIN is not valid");
            InweboUiUtils.showErrorDialog(this, R.string.pin_wrong_format);
            this.togglePinInputAccessibility(true);
            return;
        }
        this.handleSynchronizeFinalizeProcess(currentPinValue, newPinConfirmValue);
    }

    private void handleStartPasswordProcess() {
        this.togglePinInputVisibility(true);
        this.togglePinInputAccessibility(true);
    }

    private void handleSynchronizeFinalizeProcess(String currentPinCode, String newPinCode) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> this.inweboService.getApi().PwdUpdateFinalize(newPinCode, currentPinCode))
                .thenAccept(responseId -> {
                    Log.d(TAG, "PwdUpdateFinalize responded with code " + responseId);
                    this.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        Toast.makeText(
                                this,
                                this.getResources().getString(R.string.password_update_successful),
                                Toast.LENGTH_SHORT
                        ).show();

                        // On réinitialise les biokeys si besoin
                        long pinMode = super.inweboService.getApi().PinMode();
                        if ((pinMode & IW.PINMODE_BIO) != 0) {
                            resetBioKey(newPinCode);
                        } else {
                            close();
                        }

                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.startPasswordUpdateProcess();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void resetBioKey(String newPinCode) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> this.inweboService.getApi().UnsetBiokeysStart())
                .thenAccept(responseId -> {
                    Log.d(TAG, "UnsetBiokeysStart responded with code " + responseId);
                    this.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.inweboService.getApi().UnsetBiokeysFinalize(newPinCode);
                    }
                    close();
                });
    }

    private void close() {
        Intent mainIntent = new Intent(this, MenuActivity.class);
        this.startActivity(mainIntent);
    }

    private void togglePinInputVisibility(boolean isVisible) {
        int visibility = isVisible ? View.VISIBLE : View.INVISIBLE;
        this.passwordUpdateCurrentPasswordTitleView.setVisibility(visibility);
        this.passwordUpdateNewPasswordTitleView.setVisibility(visibility);
        this.passwordUpdateCurrentPinInput.setVisibility(visibility);
        this.passwordUpdateNewPinInput.setVisibility(visibility);
        this.passwordUpdateNewPinConfirmationInput.setVisibility(visibility);
        this.passwordUpdateCurrentPinInput.setEnabled(isVisible);
        this.passwordUpdateNewPinInput.setEnabled(isVisible);
        this.passwordUpdateNewPinConfirmationInput.setEnabled(isVisible);
        this.passwordUpdateCurrentPinInput.setText("");
        this.passwordUpdateNewPinInput.setText("");
        this.passwordUpdateNewPinConfirmationInput.setText("");
    }

    private void togglePinInputAccessibility(boolean isEnabled) {
        this.passwordUpdateCurrentPinInput.setEnabled(isEnabled);
        this.passwordUpdateNewPinInput.setEnabled(isEnabled);
        this.passwordUpdateNewPinConfirmationInput.setEnabled(isEnabled);
        this.passwordUpdateButton.setClickable(isEnabled);
    }

    private void resetUI() {
        Log.d(TAG, "Resetting UI");
        this.togglePinInputVisibility(false);
        this.togglePinInputAccessibility(false);
    }
}