package com.inwebo.demo_android;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.service.biometric.BiometricService;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

public class BiometricOtpGeneratorActivity extends InweboActivity {
    private static final String TAG = BiometricOtpGeneratorActivity.class.getSimpleName();

    private BiometricService biometricService;

    // containers
    private LinearLayout otpGeneratedContainer;

    // inputs
    private Button otpGeneratorButton;
    private TextView otpGeneratedInput;
    private Button otpGeneratorCopyButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_biometric_otp_generator);

        this.biometricService = BiometricService.with(this);

        this.otpGeneratedContainer = this.findViewById(R.id.otp_biometric_generator_response_container);

        this.otpGeneratorButton = this.findViewById(R.id.otp_biometric_generator_button);
        this.otpGeneratedInput = this.findViewById(R.id.otp_biometric_generator_response);
        this.otpGeneratorCopyButton = this.findViewById(R.id.otp_biometric_generator_copy_button);
    }


    @Override
    protected void onStart() {
        super.onStart();
        if (!this.biometricService.alreadyHasRegisteredBiometrics()) {
            InweboUiUtils.showErrorDialogAndExitActivity(this, R.string.biometric_no_biometric_available);
            return;
        }
        this.biometricService.setOnAuthenticationSucceededListener(new BiometricService.OnBiometricAuthenticationListener() {
            @Override
            public void onBiometricAuthenticationSucceeded(String bioKey) {
                startBiometricOtpGenerationFinalize(bioKey);
            }

            @Override
            public void onBiometricAuthenticationFailed() {
                // re-starting OTP generation process
                startBiometricOtpGenerationProcess();
            }
        });
        this.otpGeneratorButton.setOnClickListener(v ->
                handleOnlineOtpGeneration()
        );
        this.otpGeneratorCopyButton.setOnClickListener((v) -> {
            ClipboardManager clipboard = (ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            ClipData clip = ClipData.newPlainText("generated_otp", this.otpGeneratedInput.getText().toString());
            clipboard.setPrimaryClip(clip);
            this.otpGeneratorCopyButton.setClickable(false);

            Toast.makeText(
                    this,
                    this.getResources().getString(R.string.otp_generator_copy_successful),
                    Toast.LENGTH_SHORT
            ).show();
        });
        this.startBiometricOtpGenerationProcess();
    }

    private void startBiometricOtpGenerationProcess() {
        Log.i(TAG, "Starting Biometric OTP Generation process");
        this.resetUi();
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().OnlineOtpStart(0))
                .thenAccept(responseId -> {
                    Log.d(TAG, "OnlineOTPStart responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.toggleInputAccessibility(true);
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void handleOnlineOtpGeneration() {
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "PIN mode is " + pinMode);

        if (pinMode == IW.PINMODE_NONE) {
            this.startBiometricOtpGenerationFinalize("");
        } else if (pinMode == IW.PINMODE_CURRENT) {
            InweboUiUtils.showErrorDialogAndExitActivity(this, R.string.otp_bio_generator_no_bio_available);
        } else if ((pinMode & IW.PINMODE_BIO) != 0) {
            this.biometricService.authenticateToDecrypt();
        }
    }

    private void startBiometricOtpGenerationFinalize(String bioKey) {
        this.toggleInputAccessibility(false);

        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().OnlineOtpFinalizeExt(0, bioKey, 1))
                .thenAccept(responseId -> {
                    Log.d(TAG, "OnlineOtpFinalizeExt responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        String otpGenerated = super.inweboService.getApi().OtpAnswerOtp();
                        if (InweboUtils.isEmpty(otpGenerated)) {
                            Log.e(TAG, "Generated OTP is empty");
                            InweboUiUtils.showErrorDialog(this, R.string.otp_generator_error);
                            this.startBiometricOtpGenerationProcess();
                            return;
                        }
                        this.otpGeneratedContainer.setVisibility(View.VISIBLE);
                        this.otpGeneratedInput.setText(otpGenerated);

                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.startBiometricOtpGenerationProcess();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void toggleInputAccessibility(boolean isEnabled) {
        this.otpGeneratorButton.setClickable(isEnabled);
        this.otpGeneratorButton.setEnabled(isEnabled);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.toggleInputAccessibility(false);
        this.otpGeneratedContainer.setVisibility(View.GONE);
        // this button will be hidden anyway
        this.otpGeneratorCopyButton.setClickable(true);
    }
}