package com.inwebo.demo_android;

import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.service.biometric.BiometricService;
import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

/**
 * BiometricRegisterActivity is the activity to register Biometrics using the BiometricService
 * If the BiometricService returns false from canAuthenticateWithBioKey, then a dialog is displayed
 * and the user is redirected to the MenuActivity
 */
public class BiometricRegisterActivity extends InweboActivity {
    private static final String TAG = BiometricRegisterActivity.class.getSimpleName();

    private BiometricService biometricService;

    // inputs
    private Button biometricRegisterButton;
    private TextView biometricRegisterTitle;
    private TextView biometricRegisterAlreadyRegisteredTitle;
    private EditText biometricRegisterPinInput;
    private TextView biometricRegisterPinTitle;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_biometric_register);

        this.biometricService = BiometricService.with(this);

        this.biometricRegisterButton = this.findViewById(R.id.biometric_register_button);
        this.biometricRegisterTitle = this.findViewById(R.id.biometric_register_title);
        this.biometricRegisterAlreadyRegisteredTitle = this.findViewById(R.id.biometric_register_already_registered_title);
        this.biometricRegisterPinInput = this.findViewById(R.id.biometric_register_pin_input);
        this.biometricRegisterPinTitle = this.findViewById(R.id.biometric_register_pin_title);
    }

    @Override
    protected void onStart() {
        super.onStart();
        // checking if Biometrics are available on the current device
        if (!this.biometricService.canAuthenticateWithBioKey()) {
            InweboUiUtils.showErrorDialogAndExitActivity(this, R.string.biometric_no_biometric_available);
            return;
        }
        if (!this.biometricService.alreadyHasRegisteredBiometrics()) {
            this.biometricRegisterAlreadyRegisteredTitle.setVisibility(View.GONE);
        }
        this.biometricRegisterTitle.setText(R.string.biometric_authentication_register);
        this.biometricRegisterButton.setText(R.string.register);
        this.biometricRegisterButton.setOnClickListener(v -> {
            this.toggleInputAccessibility(false);
            this.startSetBioKeyFinalizeProcess();
        });
        this.biometricRegisterPinInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(biometricRegisterButton)
        );
        this.startSetBioKeyProcess();
    }

    private void startSetBioKeyProcess() {
        Log.i(TAG, "Starting Set BioKey process");
        this.resetUi();
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().SetBiokeyStart())
                .thenAccept(responseId -> {
                    Log.d(TAG, "SetBiokeyStart responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.handlePinMode();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.finish();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void handlePinMode() {
        this.toggleInputAccessibility(true);
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "PIN mode is " + pinMode);

        if (pinMode == IW.PINMODE_NONE) {
            this.toggleInputVisibility(View.GONE);
        }
    }

    private void createAuthenticationEncryption(String pin) {
        Log.i(TAG, "Creating Authentication Encryption process");
        this.biometricService.setOnAuthenticationSucceededListener(
                new BiometricService.OnBiometricAuthenticationListener() {
                    @Override
                    public void onBiometricAuthenticationSucceeded(String bioKey) {
                        handleBioKeyFinalizeProcess(bioKey, pin);
                    }

                    @Override
                    public void onBiometricAuthenticationFailed() {
                        // if biometric failed, then the process is restarted
                        startSetBioKeyProcess();
                    }
                }
        );
        this.biometricService.authenticateToEncrypt();
    }

    private void startSetBioKeyFinalizeProcess() {
        Log.i(TAG, "Starting Set BioKey Finalize process");
        String pin = this.biometricRegisterPinInput.getText().toString();
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "PIN mode is " + pinMode);

        if (pinMode == IW.PINMODE_CURRENT && InweboUtils.isEmpty(pin)) {
            // checking the pin code only if necessary
            Log.d(TAG, "Activation code value is empty");
            InweboUiUtils.showErrorDialog(this, R.string.pin_empty);
            this.startSetBioKeyProcess();
            return;
        }
        this.createAuthenticationEncryption(pin);
    }

    private void handleBioKeyFinalizeProcess(String bioKey, String pin) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().SetBiokeyFinalize(bioKey, pin))
                .thenAccept(responseId -> {
                    Log.d(TAG, "SetBiokeyFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        Toast.makeText(
                                getApplicationContext(),
                                this.getResources().getString(R.string.biometric_authentication_registered),
                                Toast.LENGTH_SHORT
                        ).show();

                        Intent mainIntent = new Intent(this, MenuActivity.class);
                        this.startActivity(mainIntent);
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.startSetBioKeyProcess();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void toggleInputVisibility(int visibility) {
        this.biometricRegisterPinTitle.setVisibility(visibility);
        this.biometricRegisterPinInput.setVisibility(visibility);
    }

    private void toggleInputAccessibility(boolean isEnabled) {
        this.biometricRegisterPinInput.setEnabled(isEnabled);
        this.biometricRegisterButton.setClickable(isEnabled);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.toggleInputAccessibility(false);
        this.toggleInputVisibility(View.VISIBLE);
    }

}