package com.inwebo.demo_android.service.biometric;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Base64;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.biometric.BiometricManager;
import androidx.biometric.BiometricPrompt;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentActivity;

import com.inwebo.demo_android.R;
import com.inwebo.demo_android.service.cryptography.CryptographyException;
import com.inwebo.demo_android.service.cryptography.CryptographyManager;
import com.inwebo.demo_android.service.cryptography.EncryptedData;

import java.math.BigInteger;
import java.security.SecureRandom;

import javax.crypto.Cipher;

/**
 * BiometricService handles all the interaction with BiometricManager, the Android Biometric API
 * BiometricService is not a singleton, and should be instantiated with {@link #with(FragmentActivity)} each time you need to interact with the Android Biometric API
 */
public class BiometricService {
    private static final String TAG = BiometricService.class.getSimpleName();
    private static final String SHARED_PREFERENCE_KEY = "InWebo-mAccess";
    private static final String SHARED_PREFERENCE_BIO_KEY = "encrypted_bio_key";
    private static final String SHARED_PREFERENCE_BIO_KEY_IV = "encrypted_bio_key_iv";

    private static final int BIOMETRICS = BiometricManager.Authenticators.BIOMETRIC_STRONG;

    private final FragmentActivity activity;
    private final BiometricManager biometricManager;
    private final CryptographyManager cryptographyManager;
    private final SharedPreferences sharedPreferences;

    private OnBiometricAuthenticationListener mBiometricAuthenticationListener;

    public static BiometricService with(FragmentActivity activity) {
        return new BiometricService(activity);
    }

    private BiometricService(FragmentActivity activity) {
        this.activity = activity;
        this.biometricManager = BiometricManager.from(activity);
        this.cryptographyManager = CryptographyManager.getInstance();
        this.sharedPreferences = activity.getSharedPreferences(SHARED_PREFERENCE_KEY, Context.MODE_PRIVATE);
    }

    /**
     * Will display a Biometric Prompt
     * Will save an encrypted BioKey in the sharedPreferences on authentication success
     */
    public void authenticateToEncrypt() {
        // Initializing the BiometricPrompt in Encryption Mode
        Cipher cipher = this.cryptographyManager.getInitializedCipherForEncryption(
                this.activity.getString(R.string.biometric_secret_key),
                true
        );
        this.authenticateWithCipher(cipher, true);
    }

    /**
     * Will display a Biometric Prompt
     * Will decrypt BioKey stored in the Shared Preferences on authentication success
     */
    public void authenticateToDecrypt() {
        // Retrieving the initialisation vector stored in the SharedPreferences to initialize the Cypher
        byte[] encryptedKeyIv = Base64.decode(this.sharedPreferences.getString(SHARED_PREFERENCE_BIO_KEY_IV, ""), Base64.NO_WRAP);

        // Initializing the BiometricPrompt in Decryption Mode
        Cipher cipher = this.cryptographyManager.getInitializedCipherForDecryption(
                this.activity.getString(R.string.biometric_secret_key),
                encryptedKeyIv,
                true
        );
        this.authenticateWithCipher(cipher, false);
    }

    /**
     * @return false if no Biometrics are available on the current device
     */
    public boolean canAuthenticateWithBioKey() {
        int authorizationId = this.biometricManager.canAuthenticate(BIOMETRICS);
        boolean canAuthenticate = authorizationId == BiometricManager.BIOMETRIC_SUCCESS;
        if (!canAuthenticate) {
            Log.d(TAG, "Device can't authenticate with biometrics. Failed with error ID : " + authorizationId);
            return false;
        }
        return true;
    }

    /**
     * alreadyHasRegisteredBiometrics checks if biometrics are available in the Shared Preferences
     *
     * @return true if the encrypted BioKey and the Initialization Vector are in the Shared Preferences
     */
    public boolean alreadyHasRegisteredBiometrics() {
        String encryptedBioKey = this.sharedPreferences.getString(SHARED_PREFERENCE_BIO_KEY, "");
        String encryptedBioKeyIv = this.sharedPreferences.getString(SHARED_PREFERENCE_BIO_KEY_IV, "");
        return !"".equals(encryptedBioKey) && !"".equals(encryptedBioKeyIv);
    }

    /**
     * authenticateWithCipher is the generic method used by {@link #authenticateToEncrypt()} and {@link #authenticateToDecrypt()}
     * It handles the BiometricPrompt and BiometricPrompt.Info displaying, as well as the callbacks
     *
     * @param cipher           the Cipher used in the Biometric process
     * @param isEncryptionMode set to true if the Biometric prompt is used for encryption purposes, false otherwise
     */
    private void authenticateWithCipher(Cipher cipher, boolean isEncryptionMode) {
        // Creating a Biometric Prompt with the activity used to instantiate the BiometricService
        BiometricPrompt biometricPrompt = new BiometricPrompt(
                this.activity,
                ContextCompat.getMainExecutor(this.activity),
                new BiometricPrompt.AuthenticationCallback() {
                    @Override
                    public void onAuthenticationSucceeded(@NonNull BiometricPrompt.AuthenticationResult result) {
                        super.onAuthenticationSucceeded(result);
                        Log.d(TAG, "Biometric authentication Succeeded");
                        String bioKey;
                        if (isEncryptionMode) {
                            bioKey = encryptFromCipher(result.getCryptoObject());
                        } else {
                            bioKey = decryptFromCypher(result.getCryptoObject());
                        }
                        if (mBiometricAuthenticationListener != null) {
                            mBiometricAuthenticationListener.onBiometricAuthenticationSucceeded(bioKey);
                        }
                    }

                    @Override
                    public void onAuthenticationError(int errorCode, @NonNull CharSequence errString) {
                        super.onAuthenticationError(errorCode, errString);
                        Log.e(TAG, "Biometric authentication error : " + errString);
                        if (mBiometricAuthenticationListener != null) {
                            mBiometricAuthenticationListener.onBiometricAuthenticationFailed();
                        }
                    }

                    @Override
                    public void onAuthenticationFailed() {
                        super.onAuthenticationFailed();
                        Log.e(TAG, "Biometric authentication failed");
                        if (mBiometricAuthenticationListener != null) {
                            mBiometricAuthenticationListener.onBiometricAuthenticationFailed();
                        }
                    }
                }
        );

        // Creating a Biometric Prompt Info
        try {
            BiometricPrompt.PromptInfo promptInfo = new BiometricPrompt.PromptInfo.Builder()
                    .setTitle(this.activity.getString(R.string.biometric_authentication_title))
                    .setSubtitle(this.activity.getString(R.string.biometric_authentication_subtitle))
                    .setDescription(this.activity.getString(R.string.biometric_authentication_description))
                    .setAllowedAuthenticators(BIOMETRICS)
                    .setConfirmationRequired(false)
                    .setNegativeButtonText(this.activity.getString(R.string.cancel))
                    .build();
            // Displaying Biometric Prompt with the Cypher
            biometricPrompt.authenticate(promptInfo, new BiometricPrompt.CryptoObject(cipher));
        } catch (IllegalArgumentException e) {
            Log.e(TAG, "Illegal argument to prompt biometry", e);
        }
    }

    /**
     * encryptFromCipher is the method called on the BiometricPrompt callback on encryption purposes
     * It will generate a SecureRandom BioKey and encrypt it with the Cipher contained in the cryptoObject
     * The encrypted BioKey and its Initialisation Vector will be stored on the SharedPreferences
     *
     * @param cryptoObject the cryptoObject returned by the BiometricPrompt
     * @return the generated BioKey
     */
    @Nullable
    private String encryptFromCipher(@Nullable BiometricPrompt.CryptoObject cryptoObject) {
        if (cryptoObject == null) {
            return null;
        }
        Cipher cipher = cryptoObject.getCipher();
        if (cipher == null) {
            return null;
        }
        // Generating random BioKey
        String bioKey = generateBioKey();

        // Encrypting the random BioKey
        EncryptedData encryptedData = this.handleDataEncryption(bioKey, cipher);

        if (encryptedData == null) {
            return null;
        }

        // Storing the encrypted random BioKey and its initialisation vector in SharedPreferences
        SharedPreferences.Editor edit = this.sharedPreferences.edit();
        edit.clear();
        edit.putString(
                SHARED_PREFERENCE_BIO_KEY,
                Base64.encodeToString(encryptedData.getCipherText(), Base64.NO_WRAP)
        );
        edit.putString(
                SHARED_PREFERENCE_BIO_KEY_IV,
                Base64.encodeToString(encryptedData.getInitialisationVector(), Base64.NO_WRAP)
        );
        edit.apply();

        // Returning clear BioKey
        return bioKey;
    }

    /**
     * decryptFromCypher is the method called on the BiometricPrompt callback on decryption purposes
     * It will retrieve the encrypted BioKey from the SharedPreferences and decrypt it with the Cipher
     * contained in the cryptoObject
     * @param cryptoObject the cryptoObject returned by the BiometricPrompt
     * @return the decrypted stored BioKey
     */
    @Nullable
    private String decryptFromCypher(@Nullable BiometricPrompt.CryptoObject cryptoObject) {
        if (cryptoObject == null) {
            return null;
        }
        Cipher cipher = cryptoObject.getCipher();
        if (cipher == null) {
            return null;
        }
        // Retrieving encrypted BioKey from SharedPreferences
        String encryptedStringBioKey = this.sharedPreferences.getString(SHARED_PREFERENCE_BIO_KEY, "");
        byte[] encryptedBioKey = Base64.decode(
                encryptedStringBioKey,
                Base64.NO_WRAP
        );
        // Returning unencrypted BioKey
        return this.handleDataDecryption(encryptedBioKey, cipher);
    }

    /**
     * Handles the data encryption if exception is thrown
     * @param bioKey the BioKey to be encrypted
     * @param cipher the Encryption Data
     * @return the encrypted data or null if exception
     */
    @Nullable
    private EncryptedData handleDataEncryption(String bioKey, Cipher cipher) {
        try {
            return this.cryptographyManager.encryptData(bioKey, cipher);
        } catch (CryptographyException e) {
            Log.e(TAG, "Can't encrypt data", e);
            return null;
        }
    }

    /**
     * Handles the data decryption if exception is thrown
     * @param encryptedBioKey the encrypted BioKey
     * @param cipher the decrypted cypher
     * @return the decrypted data or null if exception is Thrown
     */
    @Nullable
    private String handleDataDecryption(byte[] encryptedBioKey, Cipher cipher) {
        try {
            return this.cryptographyManager.decryptData(encryptedBioKey, cipher);
        } catch (CryptographyException e) {
            SharedPreferences.Editor edit = this.sharedPreferences.edit();
            edit.remove(SHARED_PREFERENCE_BIO_KEY);
            edit.remove(SHARED_PREFERENCE_BIO_KEY_IV);
            edit.apply();
            Log.e(TAG, "Can't decrypt data. Biometric data were flushed from Shared Preferences", e);
            return null;
        }
    }

    private static String generateBioKey() {
        SecureRandom random = new SecureRandom();
        return new BigInteger(130, random).toString(32);
    }

    public interface OnBiometricAuthenticationListener {
        void onBiometricAuthenticationSucceeded(String bioKey);

        void onBiometricAuthenticationFailed();
    }

    /**
     * Setting a callback called on authentication result
     *
     * @param listener the listener to be implemented
     */
    public void setOnAuthenticationSucceededListener(OnBiometricAuthenticationListener listener) {
        mBiometricAuthenticationListener = listener;
    }
}
