package com.inwebo.demo_android;

import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

public class UnlockActivity extends InweboActivity {
    private static final String TAG = UnlockActivity.class.getSimpleName();

    // containers
    private LinearLayout unlockPinContainer;

    // inputs
    private EditText unlockCodeInput;
    private Button unlockCodeButton;

    private TextView unlockPinTitle;
    private EditText unlockPinInput;
    private EditText unlockPinConfirmationInput;
    private Button unlockPinButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_unlock);

        this.unlockPinContainer = this.findViewById(R.id.unlock_pin_container);

        this.unlockCodeInput = this.findViewById(R.id.unlock_code_input);
        this.unlockCodeButton = this.findViewById(R.id.unlock_code_button);

        this.unlockPinTitle = this.findViewById(R.id.unlock_pin_title);
        this.unlockPinInput = this.findViewById(R.id.unlock_pin_input);
        this.unlockPinConfirmationInput = this.findViewById(R.id.unlock_pin_confirmation_input);
        this.unlockPinButton = this.findViewById(R.id.unlock_pin_button);
    }

    @Override
    protected void onStart() {
        super.onStart();
        this.unlockPinContainer.setVisibility(View.GONE);
        this.unlockCodeButton.setOnClickListener(v -> {
            InweboUiUtils.hideKeyboard(this, v);
            validationUnlockCode();
        });
        this.unlockPinButton.setOnClickListener(view -> {
            InweboUiUtils.hideKeyboard(this, view);
            validationUnlockPin();
        });

        this.unlockCodeInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.unlockCodeButton)
        );
        this.unlockPinConfirmationInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.unlockPinButton)
        );
    }

    public void startUnlockProcess(String activationCodeValue) {
        Log.i(TAG, "Starting Unlock Process");
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().ResetStart(activationCodeValue))
                .thenAccept(responseId -> {
                    Log.d(TAG, "ResetStart responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.handleUnlockProcess();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.toggleUnlockCodeAccessibility(true);
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void handleUnlockProcess() {
        long pinMode = super.inweboService.getApi().PinMode();

        if (pinMode == IW.PINMODE_NONE) {
            this.unlockPinContainer.setVisibility(View.GONE);
            this.startUnlockFinalize();
        } else if (pinMode == IW.PINMODE_CURRENT) {
            this.unlockPinContainer.setVisibility(View.VISIBLE);
            this.unlockPinConfirmationInput.setVisibility(View.GONE);
        } else {
            this.unlockPinTitle.setText(R.string.activation_define_pin_text);
            this.unlockPinContainer.setVisibility(View.VISIBLE);
        }
    }

    private void startUnlockFinalize() {
        Log.i(TAG, "Starting Unlock Finalize Process");
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().ResetFinalize(
                        this.unlockCodeInput.getText().toString(),
                        this.unlockPinInput.getText().toString()
                ))
                .thenAccept(responseId -> {
                    Log.d(TAG, "ResetFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);

                    if (responseId == IW.ERR_OK) {
                        Toast.makeText(
                                getApplicationContext(),
                                this.getResources().getString(R.string.unlock_successful),
                                Toast.LENGTH_SHORT
                        ).show();

                        Intent mainIntent = new Intent(this, MenuActivity.class);
                        this.startActivity(mainIntent);
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.resetUi();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void validationUnlockCode() {
        String unlockCodeValue = this.unlockCodeInput.getText().toString();

        this.toggleUnlockCodeAccessibility(false);

        if (InweboUtils.isEmpty(unlockCodeValue)) {
            Log.d(TAG, "Unlock code value is empty");
            InweboUiUtils.showErrorDialog(this, R.string.unlock_code_empty);

            this.toggleUnlockCodeAccessibility(true);
            return;
        }
        this.startUnlockProcess(unlockCodeValue);
    }

    private void validationUnlockPin() {
        String pinUnlockCodeValue = this.unlockPinInput.getText().toString();
        String pinUnlockConfirmValue = this.unlockPinConfirmationInput.getText().toString();

        this.toggleUnlockPinAccessibility(false);

        if (InweboUtils.isEmpty(pinUnlockCodeValue)) {
            Log.d(TAG, "PIN value is empty");
            InweboUiUtils.showErrorDialog(this, R.string.pin_empty);

            this.toggleUnlockPinAccessibility(true);
        } else if (!InweboUtils.isPinValid(pinUnlockCodeValue)) {
            Log.d(TAG, "PIN value is not valid");
            InweboUiUtils.showErrorDialog(this, R.string.pin_wrong_format);

            this.toggleUnlockPinAccessibility(true);
        } else if (super.inweboService.getApi().PinMode() == IW.PINMODE_NEW
                && !pinUnlockCodeValue.equals(pinUnlockConfirmValue)
        ) {
            Log.d(TAG, "PIN value and PIN value confirmation are not the same");
            InweboUiUtils.showErrorDialog(this, R.string.pin_confirmation_different);
            this.toggleUnlockPinAccessibility(true);
        } else {
            this.startUnlockFinalize();
        }
    }

    private void toggleUnlockPinAccessibility(boolean isEnable) {
        this.unlockPinInput.setEnabled(isEnable);
        this.unlockPinConfirmationInput.setEnabled(isEnable);
        this.unlockPinButton.setClickable(isEnable);
    }

    private void toggleUnlockCodeAccessibility(boolean isEnable) {
        this.unlockCodeInput.setEnabled(isEnable);
        this.unlockCodeButton.setClickable(isEnable);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.unlockPinInput.setText("");
        this.unlockCodeInput.setText("");
        this.unlockPinConfirmationInput.setText("");
        this.toggleUnlockPinAccessibility(true);
        this.toggleUnlockCodeAccessibility(true);
        this.unlockPinContainer.setVisibility(View.GONE);
    }
}