package com.inwebo.demo_android;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.utils.InweboUiUtils;


public class OfflineSealActivity extends InweboActivity {
    private static final String TAG = OnlineOtpGeneratorActivity.class.getSimpleName();

    private LinearLayout responseContainer;

    // inputs
    private EditText sealGeneratorPinInput;
    private EditText sealGeneratorDataInput;
    private Button sealGeneratorButton;
    private TextView dataTitle;
    private TextView pinTitle;
    private TextView sealGeneratedText;
    private Button sealGeneratorCopyButton;

    private TextView offlineSealResponseWillExpire;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_offline_seal_generator);

        this.responseContainer = this.findViewById(R.id.seal_generator_response_container);

        this.sealGeneratorPinInput = this.findViewById(R.id.seal_generator_pin_input);
        this.sealGeneratorDataInput = this.findViewById(R.id.seal_generator_data_input);
        this.sealGeneratorButton = this.findViewById(R.id.seal_generator_button);
        this.sealGeneratedText = this.findViewById(R.id.seal_generator_response);
        this.sealGeneratorCopyButton = this.findViewById(R.id.seal_generator_copy_button);
        this.pinTitle = this.findViewById(R.id.seal_generator_pin_title);
        this.dataTitle = this.findViewById(R.id.seal_generator_data_title);
        this.offlineSealResponseWillExpire = this.findViewById(R.id.seal_offline_generator_response_will_expire);

        this.sealGeneratorPinInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.sealGeneratorButton)
        );

        this.sealGeneratorButton.setOnClickListener((v) -> {
            InweboUiUtils.hideKeyboard(this, v);
            this.startOfflineSealGenerationFinalize();
        });

        this.sealGeneratorCopyButton.setOnClickListener((v) -> {
            ClipboardManager clipboard = (ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            ClipData clip = ClipData.newPlainText("generated_seal", this.sealGeneratedText.getText().toString());
            clipboard.setPrimaryClip(clip);
            this.sealGeneratorCopyButton.setClickable(false);

            Toast.makeText(
                    this,
                    this.getResources().getString(R.string.seal_generator_copy_successful),
                    Toast.LENGTH_SHORT
            ).show();
        });
    }

    @Override
    protected void onStart() {
        super.onStart();

        this.startOfflineSealGenerationProcess();
    }

    private void checkSynchronisation() {
        Log.i(TAG, "Checking synchronisation");
        if (this.inweboService.sealShouldSynchronize()) {
            InweboUiUtils.showChoiceDialog(
                    this,
                    R.string.otp_offline_de_synchronized,
                    () -> {
                        Intent synchronize = new Intent(this, SynchronizeActivity.class);
                        this.startActivity(synchronize);
                        this.finish();
                    },
                    this::handleOfflineSealGeneration
            );
            return;
        }
        this.handleOfflineSealGeneration();
    }

    private void startOfflineSealGenerationProcess() {
        Log.i(TAG, "Starting Offline Seal Generation process");
        this.resetUi();
        this.checkSynchronisation();
    }

    private void startOfflineSealGenerationFinalize() {
        this.toggleInputAccessibility(false);
        String pin = this.sealGeneratorPinInput.getText().toString();
        String data = this.sealGeneratorDataInput.getText().toString();
        String sealGenerated = this.inweboService.getApi().SealGenerate(pin, data);
        this.sealGeneratedText.setText(sealGenerated);
        this.responseContainer.setVisibility(View.VISIBLE);

        long secondsAvailable = this.inweboService.getApi().DisplayTime();

        new CountDownTimer(secondsAvailable * 1000, 1000) {

            @Override
            public void onTick(long millisUntilFinished) {
                long remainingSeconds = millisUntilFinished / 1000;
                offlineSealResponseWillExpire.setText(
                        getResources().getString(R.string.otp_offline_generator_will_expire, remainingSeconds)
                );
            }

            @Override
            public void onFinish() {
                setSealToExpired();
            }
        }.start();
    }

    private void setSealToExpired() {
        this.offlineSealResponseWillExpire.setText(R.string.otp_offline_generator_has_expired);
        this.sealGeneratorCopyButton.setEnabled(false);
    }

    private void handleOfflineSealGeneration() {
        Log.i(TAG, "Starting offline Seal generation");
        boolean pinRequired = this.inweboService.getApi().SealModeQuery(0) == 1;

        if (pinRequired) {
            this.toggleInputVisibility(true);
            this.toggleInputAccessibility(true);
        }
    }

    private void toggleInputVisibility(boolean isVisible) {
        int visibility = isVisible ? View.VISIBLE : View.INVISIBLE;
        this.pinTitle.setVisibility(visibility);
        this.dataTitle.setVisibility(visibility);
        this.sealGeneratorButton.setVisibility(visibility);
        this.sealGeneratorDataInput.setVisibility(visibility);
        this.sealGeneratorDataInput.setText("");
        this.sealGeneratorPinInput.setVisibility(visibility);
        this.sealGeneratorPinInput.setText("");
    }

    private void toggleInputAccessibility(boolean isEnabled) {
        this.sealGeneratorPinInput.setEnabled(isEnabled);
        this.sealGeneratorDataInput.setEnabled(isEnabled);
        this.sealGeneratorButton.setClickable(isEnabled);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.toggleInputAccessibility(false);
        this.toggleInputVisibility(false);
        this.responseContainer.setVisibility(View.GONE);
        // this button will be hidden anyway
        this.sealGeneratorCopyButton.setClickable(true);
    }
}
