package com.inwebo.demo_android;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.utils.InweboUiUtils;

public class OfflineOtpGeneratorActivity extends InweboActivity {
    private static final String TAG = OfflineOtpGeneratorActivity.class.getSimpleName();

    // containers
    private LinearLayout responseContainer;

    // inputs
    private TextView offlineOtpPinTitle;
    private EditText offlineOtpPinInput;
    private Button offlineOtpPinButton;
    private TextView offlineOtpResponseInput;
    private Button offlineOtpPinCopyButton;

    private TextView offlineOtpResponseWillExpire;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_offline_otp_generator);

        this.responseContainer = this.findViewById(R.id.otp_offline_generator_response_container);

        this.offlineOtpPinTitle = this.findViewById(R.id.otp_offline_generator_pin_title);
        this.offlineOtpPinInput = this.findViewById(R.id.otp_offline_generator_pin_input);
        this.offlineOtpPinButton = this.findViewById(R.id.otp_offline_generator_button);
        this.offlineOtpResponseInput = this.findViewById(R.id.otp_offline_generator_response_input);
        this.offlineOtpPinCopyButton = this.findViewById(R.id.otp_offline_generator_copy_button);
        this.offlineOtpResponseWillExpire = this.findViewById(R.id.otp_offline_generator_response_will_expire);
    }

    @Override
    protected void onStart() {
        super.onStart();
        this.offlineOtpResponseInput.setEnabled(false);
        this.offlineOtpPinInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.offlineOtpPinButton)
        );
        this.offlineOtpPinButton.setOnClickListener((v) -> {
            InweboUiUtils.hideKeyboard(this, v);
            this.togglePinInputAccessibility(false);
            this.startOtpGenerateFinalize();
        });
        this.offlineOtpPinCopyButton.setOnClickListener((v) -> {
            ClipboardManager clipboard = (ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            ClipData clip = ClipData.newPlainText("generated_offline_otp", this.offlineOtpResponseInput.getText().toString());
            clipboard.setPrimaryClip(clip);
            this.offlineOtpPinCopyButton.setClickable(false);

            Toast.makeText(
                    this,
                    this.getResources().getString(R.string.otp_generator_copy_successful),
                    Toast.LENGTH_SHORT
            ).show();
        });
        this.startOtpGenerationProcess();
    }

    private void startOtpGenerationProcess() {
        this.resetUI();
        this.checkSynchronisation();
    }

    private void checkSynchronisation() {
        Log.i(TAG, "Checking synchronisation");
        if (this.inweboService.otpShouldSynchronize()) {
            InweboUiUtils.showChoiceDialog(
                    this,
                    R.string.otp_offline_de_synchronized,
                    () -> {
                        Intent synchronize = new Intent(this, SynchronizeActivity.class);
                        this.startActivity(synchronize);
                        this.finish();
                    },
                    this::handleOtpGenerate
            );
            return;
        }
        this.handleOtpGenerate();
    }

    private void handleOtpGenerate() {
        Log.i(TAG, "Starting offline OTP generation");
        boolean pinRequired = this.inweboService.getApi().OtpModeQuery(0) == 1;
        this.togglePinInputAccessibility(true);
        if (pinRequired) {
            this.togglePinInputVisibility(true);
        }
    }

    private void startOtpGenerateFinalize() {
        Log.i(TAG, "Starting offline OTP generation finalisation");
        String pin = this.offlineOtpPinInput.getText().toString();
        String otpGenerated = this.inweboService.getApi().OtpGenerate(pin);
        this.offlineOtpResponseInput.setText(otpGenerated);
        this.toggleResponseContainerVisibility(true);

        long secondsAvailable = this.inweboService.getApi().DisplayTime();

        new CountDownTimer(secondsAvailable * 1000, 1000) {

            @Override
            public void onTick(long millisUntilFinished) {
                long remainingSeconds = millisUntilFinished / 1000;
                offlineOtpResponseWillExpire.setText(
                        getResources().getString(R.string.otp_offline_generator_will_expire, remainingSeconds)
                );
            }

            @Override
            public void onFinish() {
                setOtpToExpired();
            }
        }.start();
    }

    private void setOtpToExpired() {
        this.offlineOtpResponseWillExpire.setText(R.string.otp_offline_generator_has_expired);
        this.offlineOtpPinCopyButton.setEnabled(false);
    }

    private void togglePinInputVisibility(boolean isVisible) {
        int visibility = isVisible ? View.VISIBLE : View.INVISIBLE;
        this.offlineOtpPinTitle.setVisibility(visibility);
        this.offlineOtpPinInput.setVisibility(visibility);
    }

    private void togglePinInputAccessibility(boolean isEnable) {
        this.offlineOtpPinInput.setEnabled(isEnable);
        this.offlineOtpPinButton.setEnabled(isEnable);
    }

    private void toggleResponseContainerVisibility(boolean isVisible) {
        int visibility = isVisible ? View.VISIBLE : View.INVISIBLE;
        this.responseContainer.setVisibility(visibility);
    }

    private void resetUI() {
        Log.d(TAG, "Resetting UI");
        this.togglePinInputVisibility(false);
        this.togglePinInputAccessibility(false);
        this.toggleResponseContainerVisibility(false);
    }
}