package com.inwebo.demo_android;

import android.app.AlertDialog;
import android.os.Bundle;
import android.text.InputType;
import android.text.method.PasswordTransformationMethod;
import android.util.Log;
import android.view.WindowManager;
import android.widget.EditText;

import androidx.appcompat.app.AppCompatActivity;
import androidx.lifecycle.MutableLiveData;
import androidx.lifecycle.Observer;

import com.google.firebase.messaging.FirebaseMessaging;
import com.inwebo.demo_android.firebase.IWFirebaseMessagingService;
import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.service.InweboService;
import com.inwebo.demo_android.service.biometric.BiometricService;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

import java.util.Map;

public class InweboActivity extends AppCompatActivity {

    protected InweboService inweboService;

    private BiometricService biometricService;

    // InAppMessage
    protected MutableLiveData<Map<String, String>> inAppMessage;
    protected MutableLiveData<String> fcmToken;

    private static final String PUSH_TAG = InweboActivity.class.getSimpleName();


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        this.biometricService = BiometricService.with(this);

        // prevent from screenshots
        super.getWindow().setFlags(WindowManager.LayoutParams.FLAG_SECURE, WindowManager.LayoutParams.FLAG_SECURE);

        this.inweboService = InweboService.with(this);

        setupInAppMessageObserver();
        setupFCMTokenChangeObserver();
    }

    @Override
    protected void onResume() {
        super.onResume();
        if (InweboUtils.isDeviceRooted()) {
            InweboUiUtils.showAlertDialogAndExitApp(this, R.string.phone_rooted);
        }
    }

    /*
    ** Setup Firebase Token update
    */
    private void setupFCMTokenChangeObserver() {
        this.fcmToken = IWFirebaseMessagingService.getNewToken();

        final Observer<String> tokenObserver = token -> {
            if (token != null) {
                registerForPushNotification();
                fcmToken.postValue(null);
            }
        };

        fcmToken.observe(this, tokenObserver);
    }

    private void registerForPushNotification() {
        FirebaseMessaging.getInstance().getToken()
                .addOnCompleteListener(task -> {
                    if (!task.isSuccessful()) {
                        Log.w(PUSH_TAG, "Fetching FCM registration token failed", task.getException());
                        return;
                    }

                    // Get new FCM registration token
                    final String token = task.getResult();

                    InweboActivity.this.inweboService.getApi().SetDeviceOS("firebase");

                    Log.i(PUSH_TAG, "Starting push registration");
                    InweboPromise.with(InweboActivity.this)
                            .fulfillInAsync(() ->
                                    InweboActivity.this.inweboService.getApi().PushRegistrationStart()
                            )
                            .thenAccept(responseId -> {
                                Log.d(PUSH_TAG, "PushRegistrationStart responded with code " + responseId);
                                InweboActivity.this.inweboService.setDataToInternalStorage(InweboActivity.this);
                                if (responseId == IW.ERR_OK) {
                                    finalizePushRegistration(token);
                                }
                            })
                            .onError(throwable -> {
                                Log.e(PUSH_TAG, "Error during promise execution", throwable);
                            });
                });
    }

    private void finalizePushRegistration(String fcmToken) {
        Log.i(PUSH_TAG, "Finalize push registration");
        InweboPromise.with(this)
                .fulfillInAsync(() -> this.inweboService.getApi().PushRegistrationFinalize(fcmToken))
                .thenAccept(responseId -> {
                    Log.d(PUSH_TAG, "PushRegistrationFinalize responded with code " + responseId);
                    this.inweboService.setDataToInternalStorage(this);
                })
                .onError(throwable -> {
                    Log.e(PUSH_TAG, "Error during promise execution", throwable);
                });
    }


    /*
    ** Setup to respond to inApp authentication request
    */
    private void setupInAppMessageObserver() {
        this.inAppMessage = IWFirebaseMessagingService.getInAppMessage();

        final Observer<Map<String, String>> messageObserver = message -> {
            if (message != null) {
                String serviceName = message.get("serviceName");
                String alias = message.get("alias");
                if ("authenticate".equals(message.get("action"))) {
                    showPushOTPAuthentication(serviceName, alias);
                } else {
                    String text = InweboUtils.mapToFormattedString(message);
                    InweboUiUtils.showInfoDialog(this, getString(R.string.notification_title), text);
                }
                inAppMessage.postValue(null);
            }
        };

        inAppMessage.observe(this, messageObserver);
    }

    protected void showPushOTPAuthentication(String serviceName, String alias) {
        String message = (serviceName == null)
                ? getString(R.string.notification_otp)
                : getString(R.string.notification_otp_format, serviceName);
        InweboUiUtils.showChoiceDialog(this, getString(R.string.notification_title), message,
                () -> {
                    // YES
                    sendPushOTP(serviceName, alias);
                },
                () -> {
                    // NO
                    sendOTP(alias, "", false, false);
                });
    }

    private void sendPushOTP(String serviceName, String alias) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> this.inweboService.getApi().PushOtpStart(alias))
                .thenAccept(responseId -> {
                    Log.d(PUSH_TAG, "PushOtpStart responded with code " + responseId);
                    this.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        long pinMode = this.inweboService.getApi().PinMode();
                        if (pinMode == IW.PINMODE_NONE) {
                            sendPushOTPNoPin(alias);
                        } else if (pinMode == IW.PINMODE_CURRENT) {
                            sendPushOTPPin(alias);
                        } else if ((pinMode & IW.PINMODE_BIO) != 0) {
                            sendPushOTPBio(serviceName, alias);
                        }
                    } else {
                        InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    }
                });
    }

    private void sendPushOTPBio(String serviceName, String alias) {
        this.biometricService.setOnAuthenticationSucceededListener(new BiometricService.OnBiometricAuthenticationListener() {
            @Override
            public void onBiometricAuthenticationSucceeded(String bioKey) {
                sendOTP(alias, bioKey, true, true);
            }

            @Override
            public void onBiometricAuthenticationFailed() {
                // re-starting OTP generation process
                showPushOTPAuthentication(serviceName, alias);
            }
        });
    }

    private void sendPushOTPNoPin(String alias) {
        sendOTP(alias, "", true, false);
    }

    private void sendPushOTPPin(String alias) {
        AlertDialog.Builder alert = new AlertDialog.Builder(this);

        final EditText edittext = new EditText(this);
        edittext.setTransformationMethod(PasswordTransformationMethod.getInstance());
        edittext.setInputType(InputType.TYPE_CLASS_NUMBER | InputType.TYPE_NUMBER_VARIATION_PASSWORD);
        alert.setTitle(getString(R.string.notification_otp_pin));

        alert.setView(edittext);

        alert.setPositiveButton(getString(R.string.notification_otp_pin_button), (dialog, whichButton) -> {
            String pin = edittext.getText().toString();
            sendOTP(alias, pin, true, false);
        });

        alert.show();
    }

    private void sendOTP(String alias, String pin, Boolean ok, Boolean isBio) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> {
                    if ("".equals(pin)) {
                        return this.inweboService.getApi().PushOtpFinalize(alias, pin, ok);
                    }
                    return this.inweboService.getApi().PushOtpFinalizeExt(alias, pin, ok, isBio ? 1 : 0);
                })
                .thenAccept(responseId -> {
                    Log.d(PUSH_TAG, "PushOTPFinalize responded with code " + responseId);
                    this.inweboService.setDataToInternalStorage(this);
                    if (responseId != IW.ERR_OK) {
                        InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    }
                });
    }
}