package com.inwebo.demo_android;

import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

/**
 * SynchronizeActivity implements the Synchronization process
 */
public class SynchronizeActivity extends InweboActivity {
    private static final String TAG = SynchronizeActivity.class.getSimpleName();

    private TextView synchronizePinTitleView;
    private EditText synchronizePinInput;
    private Button synchronizePinButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_synchronize);

        this.synchronizePinTitleView = this.findViewById(R.id.synchronize_pin_title);
        this.synchronizePinInput = this.findViewById(R.id.synchronize_pin_input);
        this.synchronizePinButton = this.findViewById(R.id.synchronize_pin_button);
    }

    @Override
    protected void onStart() {
        super.onStart();
        this.synchronizePinInput.setOnEditorActionListener(
                InweboUiUtils.performClickOnUserBlur(this.synchronizePinButton)
        );
        this.synchronizePinButton.setOnClickListener(v -> {
            InweboUiUtils.hideKeyboard(this, v);
            this.startSynchronizeFinalize();
        });

        this.startSynchronizeProcess();
    }

    private void startSynchronizeProcess() {
        Log.i(TAG, "Starting the Synchronize process");
        this.resetUI();
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().SynchronizeStart())
                .thenAccept(responseId -> {
                    Log.d(TAG, "SynchronizeStart responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);

                    if (responseId == IW.ERR_OK) {
                        this.handleSynchronizeProcess();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.finish();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void handleSynchronizeProcess() {
        long pinMode = super.inweboService.getApi().PinMode();
        if (pinMode == IW.PINMODE_NONE) {
            // already signed in, no need to enter pin
            this.togglePinInputVisibility(false);
            synchronizePinButton.setVisibility(View.VISIBLE);
            synchronizePinButton.setClickable(true);

        } else if (pinMode == IW.PINMODE_CURRENT) {
            this.togglePinInputVisibility(true);
            this.togglePinInputAccessibility(true);
        }
    }

    private void startSynchronizeFinalize() {
        Log.i(TAG, "Starting Synchronize Finalize process");

        this.synchronizePinInput.setEnabled(false);
        this.synchronizePinButton.setClickable(false);
        String pinValue = this.synchronizePinInput.getText().toString();
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "Pin mode is " + pinMode);
        if (pinMode == IW.PINMODE_NONE) {
            handleSynchronizeFinalizeProcess("");
        } else if (pinMode == IW.PINMODE_CURRENT) {
            if (InweboUtils.isEmpty(pinValue)) {
                Log.d(TAG, "PIN value is Empty");
                InweboUiUtils.showErrorDialog(this, R.string.pin_empty);
                this.startSynchronizeProcess();
                return;
            }
            this.handleSynchronizeFinalizeProcess(pinValue);
        }
    }

    private void handleSynchronizeFinalizeProcess(String pinCode) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().SynchronizeFinalize(pinCode))
                .thenAccept((responseId) -> {
                    Log.d(TAG, "SynchronizeFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);

                    if (responseId == IW.ERR_OK) {
                        Toast.makeText(
                                getApplicationContext(),
                                this.getResources().getString(R.string.synchronize_successful),
                                Toast.LENGTH_SHORT
                        ).show();

                        Intent mainIntent = new Intent(this, MenuActivity.class);
                        this.startActivity(mainIntent);
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.startSynchronizeProcess();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void togglePinInputVisibility(boolean isVisible) {
        int visibility = isVisible ? View.VISIBLE : View.INVISIBLE;
        this.synchronizePinTitleView.setVisibility(visibility);
        this.synchronizePinInput.setVisibility(visibility);
        this.synchronizePinInput.setEnabled(isVisible);
        this.synchronizePinInput.setText("");
    }

    private void togglePinInputAccessibility(boolean isEnabled) {
        this.synchronizePinInput.setEnabled(isEnabled);
        this.synchronizePinButton.setClickable(isEnabled);
    }

    private void resetUI() {
        Log.d(TAG, "Resetting UI");
        this.togglePinInputVisibility(false);
        this.togglePinInputAccessibility(false);
    }
}