package com.inwebo.demo_android;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.inwebo.demo_android.network.InweboPromise;
import com.inwebo.demo_android.utils.InweboUiUtils;
import com.inwebo.demo_android.utils.InweboUtils;
import com.inwebo.iwlib.IW;

public class OnlineSealActivity extends InweboActivity {
    private static final String TAG = OnlineOtpGeneratorActivity.class.getSimpleName();

    private LinearLayout responseContainer;

    // inputs
    private EditText sealGeneratorPinInput;
    private EditText sealGeneratorDataInput;
    private Button sealGeneratorButton;
    private TextView dataTitle;
    private TextView pinTitle;
    private TextView sealGeneratedText;
    private Button sealGeneratorCopyButton;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_online_seal_generator);

        this.responseContainer = this.findViewById(R.id.seal_generator_response_container);

        this.sealGeneratorPinInput = this.findViewById(R.id.seal_generator_pin_input);
        this.sealGeneratorDataInput = this.findViewById(R.id.seal_generator_data_input);
        this.sealGeneratorButton = this.findViewById(R.id.seal_generator_button);
        this.sealGeneratedText = this.findViewById(R.id.seal_generator_response);
        this.sealGeneratorCopyButton = this.findViewById(R.id.seal_generator_copy_button);
        this.pinTitle = this.findViewById(R.id.seal_generator_pin_title);
        this.dataTitle = this.findViewById(R.id.seal_generator_data_title);

        this.sealGeneratorPinInput.setOnEditorActionListener(
            InweboUiUtils.performClickOnUserBlur(this.sealGeneratorButton)
        );

        this.sealGeneratorButton.setOnClickListener((v) -> {
            InweboUiUtils.hideKeyboard(this, v);
            this.startOnlineSealGenerationFinalize();
        });

        this.sealGeneratorCopyButton.setOnClickListener((v) -> {
            ClipboardManager clipboard = (ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            ClipData clip = ClipData.newPlainText("generated_seal", this.sealGeneratedText.getText().toString());
            clipboard.setPrimaryClip(clip);
            this.sealGeneratorCopyButton.setClickable(false);

            Toast.makeText(
                    this,
                    this.getResources().getString(R.string.seal_generator_copy_successful),
                    Toast.LENGTH_SHORT
            ).show();
        });
    }

    @Override
    protected void onStart() {
        super.onStart();

        this.startOnlineSealGenerationProcess();
    }

    private void startOnlineSealGenerationProcess() {
        Log.i(TAG, "Starting Online Seal Generation process");
        this.resetUi();
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().OnlineSealStart(0))
                .thenAccept(responseId -> {
                    Log.d(TAG, "OnlineSealStart responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        this.handleOnlineSealGeneration();
                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.finish();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void startOnlineSealGenerationFinalize() {
        this.toggleInputAccessibility(false);
        String pin = this.sealGeneratorPinInput.getText().toString();
        String data = this.sealGeneratorDataInput.getText().toString();
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "PIN mode is " + pinMode);

        if (pinMode == IW.PINMODE_NONE) {
            this.handleOnlineSealGenerationFinalize("", data);
        } else if ((pinMode & IW.PINMODE_CURRENT) != 0) {
            if (InweboUtils.isEmpty(pin)) {
                Log.d(TAG, "Activation code value is empty");
                InweboUiUtils.showErrorDialog(this, R.string.pin_empty);
                this.startOnlineSealGenerationProcess();
                return;
            }
            this.handleOnlineSealGenerationFinalize(pin, data);
        }
    }

    private void handleOnlineSealGenerationFinalize(String pin, String data) {
        InweboPromise.with(this)
                .fulfillInAsync(() -> super.inweboService.getApi().OnlineSealFinalize(0, pin, data))
                .thenAccept(responseId -> {
                    Log.d(TAG, "OnlineSealFinalize responded with code " + responseId);
                    super.inweboService.setDataToInternalStorage(this);
                    if (responseId == IW.ERR_OK) {
                        String sealGenerated = super.inweboService.getApi().SealAnswerOtp();
                        if (InweboUtils.isEmpty(sealGenerated)) {
                            Log.e(TAG, "Generated Seal is empty");
                            InweboUiUtils.showErrorDialog(this, R.string.seal_generator_error);
                            this.startOnlineSealGenerationProcess();
                            return;
                        }
                        this.responseContainer.setVisibility(View.VISIBLE);
                        this.sealGeneratedText.setText(sealGenerated);

                        return;
                    }
                    InweboUiUtils.showErrorDialogFromResponseId(this, responseId);
                    this.startOnlineSealGenerationProcess();
                })
                .onError(throwable -> {
                    Log.e(TAG, "Error during promise execution", throwable);
                    InweboUiUtils.showNetworkErrorDialog(this);
                });
    }

    private void handleOnlineSealGeneration() {
        long pinMode = super.inweboService.getApi().PinMode();

        Log.d(TAG, "PIN mode is " + pinMode);

        if ((pinMode & IW.PINMODE_CURRENT) != 0) {
            this.toggleInputVisibility(true);
            this.toggleInputAccessibility(true);
        }
    }

    private void toggleInputVisibility(boolean isVisible) {
        int visibility = isVisible ? View.VISIBLE : View.INVISIBLE;
        this.pinTitle.setVisibility(visibility);
        this.dataTitle.setVisibility(visibility);
        this.sealGeneratorButton.setVisibility(visibility);
        this.sealGeneratorDataInput.setVisibility(visibility);
        this.sealGeneratorDataInput.setText("");
        this.sealGeneratorPinInput.setVisibility(visibility);
        this.sealGeneratorPinInput.setText("");
    }

    private void toggleInputAccessibility(boolean isEnabled) {
        this.sealGeneratorPinInput.setEnabled(isEnabled);
        this.sealGeneratorDataInput.setEnabled(isEnabled);
        this.sealGeneratorButton.setClickable(isEnabled);
    }

    private void resetUi() {
        Log.d(TAG, "Resetting UI");
        this.toggleInputAccessibility(false);
        this.toggleInputVisibility(false);
        this.responseContainer.setVisibility(View.GONE);
        // this button will be hidden anyway
        this.sealGeneratorCopyButton.setClickable(true);
    }
}
